import { Task } from '../task';
export type Config = {
    host: string;
    apiKey?: string;
    clientAgents?: string[];
    requestConfig?: Partial<Omit<RequestInit, 'body' | 'method'>>;
    httpClient?: (input: string, init?: RequestInit) => Promise<any>;
    timeout?: number;
};
export type Pagination = {
    offset?: number;
    limit?: number;
};
export type ResourceQuery = Pagination & {};
export type ResourceResults<T> = Pagination & {
    results: T;
    total: number;
};
export type IndexOptions = {
    primaryKey?: string;
};
export type IndexObject = {
    uid: string;
    primaryKey?: string;
    createdAt: Date;
    updatedAt: Date;
};
export type IndexesQuery = ResourceQuery & {};
export type IndexesResults<T> = ResourceResults<T> & {};
export declare const MatchingStrategies: {
    readonly ALL: "all";
    readonly LAST: "last";
};
export type MatchingStrategies = (typeof MatchingStrategies)[keyof typeof MatchingStrategies];
export type Filter = string | Array<string | string[]>;
export type Query = {
    q?: string | null;
};
export type Highlight = {
    attributesToHighlight?: string[];
    highlightPreTag?: string;
    highlightPostTag?: string;
};
export type Crop = {
    attributesToCrop?: string[];
    cropLength?: number;
    cropMarker?: string;
};
export type SearchForFacetValuesParams = Omit<SearchParams, 'facetName'> & {
    facetName: string;
};
export type FacetHit = {
    value: string;
    count: number;
};
export type SearchForFacetValuesResponse = {
    facetHits: FacetHit[];
    facetQuery: string | null;
    processingTimeMs: number;
};
export type HybridSearch = {
    embedder?: string;
    semanticRatio?: number;
};
export type SearchParams = Query & Pagination & Highlight & Crop & {
    filter?: Filter;
    sort?: string[];
    facets?: string[];
    attributesToRetrieve?: string[];
    showMatchesPosition?: boolean;
    matchingStrategy?: MatchingStrategies;
    hitsPerPage?: number;
    page?: number;
    facetName?: string;
    facetQuery?: string;
    vector?: number[] | null;
    showRankingScore?: boolean;
    showRankingScoreDetails?: boolean;
    attributesToSearchOn?: string[] | null;
    hybrid?: HybridSearch;
};
export type SearchRequestGET = Pagination & Query & Omit<Highlight, 'attributesToHighlight'> & Omit<Crop, 'attributesToCrop'> & {
    filter?: string;
    sort?: string;
    facets?: string;
    attributesToRetrieve?: string;
    attributesToHighlight?: string;
    attributesToCrop?: string;
    showMatchesPosition?: boolean;
    vector?: string | null;
    attributesToSearchOn?: string | null;
    hybridEmbedder?: string;
    hybridSemanticRatio?: number;
};
export type MultiSearchQuery = SearchParams & {
    indexUid: string;
};
export type MultiSearchParams = {
    queries: MultiSearchQuery[];
};
export type CategoriesDistribution = {
    [category: string]: number;
};
export type Facet = string;
export type FacetDistribution = Record<Facet, CategoriesDistribution>;
export type MatchesPosition<T> = Partial<Record<keyof T, Array<{
    start: number;
    length: number;
}>>>;
export type Hit<T = Record<string, any>> = T & {
    _formatted?: Partial<T>;
    _matchesPosition?: MatchesPosition<T>;
    _rankingScore?: number;
    _rankingScoreDetails?: RankingScoreDetails;
};
export type RankingScoreDetails = {
    words?: {
        order: number;
        matchingWords: number;
        maxMatchingWords: number;
        score: number;
    };
    typo?: {
        order: number;
        typoCount: number;
        maxTypoCount: number;
        score: number;
    };
    proximity?: {
        order: number;
        score: number;
    };
    attribute?: {
        order: number;
        attributes_ranking_order: number;
        attributes_query_word_order: number;
        score: number;
    };
    exactness?: {
        order: number;
        matchType: string;
        score: number;
    };
    [key: string]: Record<string, any> | undefined;
};
export type Hits<T = Record<string, any>> = Array<Hit<T>>;
export type FacetStat = {
    min: number;
    max: number;
};
export type FacetStats = Record<string, FacetStat>;
export type SearchResponse<T = Record<string, any>, S extends SearchParams | undefined = undefined> = {
    hits: Hits<T>;
    processingTimeMs: number;
    query: string;
    facetDistribution?: FacetDistribution;
    facetStats?: FacetStats;
} & (undefined extends S ? Partial<FinitePagination & InfinitePagination> : true extends IsFinitePagination<NonNullable<S>> ? FinitePagination : InfinitePagination);
type FinitePagination = {
    totalHits: number;
    hitsPerPage: number;
    page: number;
    totalPages: number;
};
type InfinitePagination = {
    offset: number;
    limit: number;
    estimatedTotalHits: number;
};
type IsFinitePagination<S extends SearchParams> = Or<HasHitsPerPage<S>, HasPage<S>>;
type Or<A extends boolean, B extends boolean> = true extends A ? true : true extends B ? true : false;
type HasHitsPerPage<S extends SearchParams> = undefined extends S['hitsPerPage'] ? false : true;
type HasPage<S extends SearchParams> = undefined extends S['page'] ? false : true;
export type MultiSearchResult<T> = SearchResponse<T> & {
    indexUid: string;
};
export type MultiSearchResponse<T = Record<string, any>> = {
    results: Array<MultiSearchResult<T>>;
};
export type FieldDistribution = {
    [field: string]: number;
};
type Fields<T = Record<string, any>> = Array<Extract<keyof T, string>> | Extract<keyof T, string>;
export type DocumentOptions = {
    primaryKey?: string;
};
export declare const ContentTypeEnum: Readonly<Record<string, ContentType>>;
export type ContentType = 'text/csv' | 'application/x-ndjson' | 'application/json';
export type RawDocumentAdditionOptions = DocumentOptions & {
    csvDelimiter?: string;
};
export type DocumentsQuery<T = Record<string, any>> = ResourceQuery & {
    fields?: Fields<T>;
    filter?: Filter;
    limit?: number;
    offset?: number;
};
export type DocumentQuery<T = Record<string, any>> = {
    fields?: Fields<T>;
};
export type DocumentsDeletionQuery = {
    filter: Filter;
};
export type DocumentsIds = string[] | number[];
export type FilterableAttributes = string[] | null;
export type DistinctAttribute = string | null;
export type SearchableAttributes = string[] | null;
export type SortableAttributes = string[] | null;
export type DisplayedAttributes = string[] | null;
export type RankingRules = string[] | null;
export type StopWords = string[] | null;
export type Synonyms = {
    [field: string]: string[];
} | null;
export type TypoTolerance = {
    enabled?: boolean | null;
    disableOnAttributes?: string[] | null;
    disableOnWords?: string[] | null;
    minWordSizeForTypos?: {
        oneTypo?: number | null;
        twoTypos?: number | null;
    };
} | null;
export type SeparatorTokens = string[] | null;
export type NonSeparatorTokens = string[] | null;
export type Dictionary = string[] | null;
export type ProximityPrecision = 'byWord' | 'byAttribute';
export type Distribution = {
    mean: number;
    sigma: number;
};
export type OpenAiEmbedder = {
    source: 'openAi';
    model?: string;
    apiKey?: string;
    documentTemplate?: string;
    dimensions?: number;
    distribution?: Distribution;
};
export type HuggingFaceEmbedder = {
    source: 'huggingFace';
    model?: string;
    revision?: string;
    documentTemplate?: string;
    distribution?: Distribution;
};
export type UserProvidedEmbedder = {
    source: 'userProvided';
    dimensions: number;
    distribution?: Distribution;
};
export type RestEmbedder = {
    source: 'rest';
    url: string;
    apiKey?: string;
    dimensions?: number;
    documentTemplate?: string;
    inputField?: string[] | null;
    inputType?: 'text' | 'textArray';
    query?: Record<string, any> | null;
    pathToEmbeddings?: string[] | null;
    embeddingObject?: string[] | null;
    distribution?: Distribution;
};
export type OllamaEmbedder = {
    source: 'ollama';
    url?: string;
    apiKey?: string;
    model?: string;
    documentTemplate?: string;
    distribution?: Distribution;
};
export type Embedder = OpenAiEmbedder | HuggingFaceEmbedder | UserProvidedEmbedder | RestEmbedder | OllamaEmbedder | null;
export type Embedders = Record<string, Embedder> | null;
export type FacetOrder = 'alpha' | 'count';
export type Faceting = {
    maxValuesPerFacet?: number | null;
    sortFacetValuesBy?: Record<string, FacetOrder> | null;
};
export type PaginationSettings = {
    maxTotalHits?: number | null;
};
export type SearchCutoffMs = number | null;
export type Settings = {
    filterableAttributes?: FilterableAttributes;
    distinctAttribute?: DistinctAttribute;
    sortableAttributes?: SortableAttributes;
    searchableAttributes?: SearchableAttributes;
    displayedAttributes?: DisplayedAttributes;
    rankingRules?: RankingRules;
    stopWords?: StopWords;
    synonyms?: Synonyms;
    typoTolerance?: TypoTolerance;
    faceting?: Faceting;
    pagination?: PaginationSettings;
    separatorTokens?: SeparatorTokens;
    nonSeparatorTokens?: NonSeparatorTokens;
    dictionary?: Dictionary;
    proximityPrecision?: ProximityPrecision;
    embedders?: Embedders;
    searchCutoffMs?: SearchCutoffMs;
};
export declare const TaskStatus: {
    readonly TASK_SUCCEEDED: "succeeded";
    readonly TASK_PROCESSING: "processing";
    readonly TASK_FAILED: "failed";
    readonly TASK_ENQUEUED: "enqueued";
    readonly TASK_CANCELED: "canceled";
};
export type TaskStatus = (typeof TaskStatus)[keyof typeof TaskStatus];
export declare const TaskTypes: {
    readonly DOCUMENTS_ADDITION_OR_UPDATE: "documentAdditionOrUpdate";
    readonly DOCUMENT_DELETION: "documentDeletion";
    readonly DUMP_CREATION: "dumpCreation";
    readonly INDEX_CREATION: "indexCreation";
    readonly INDEX_DELETION: "indexDeletion";
    readonly INDEXES_SWAP: "indexSwap";
    readonly INDEX_UPDATE: "indexUpdate";
    readonly SETTINGS_UPDATE: "settingsUpdate";
    readonly SNAPSHOT_CREATION: "snapshotCreation";
    readonly TASK_CANCELATION: "taskCancelation";
    readonly TASK_DELETION: "taskDeletion";
};
export type TaskTypes = (typeof TaskTypes)[keyof typeof TaskTypes];
export type TasksQuery = {
    indexUids?: string[];
    uids?: number[];
    types?: TaskTypes[];
    statuses?: TaskStatus[];
    canceledBy?: number[];
    beforeEnqueuedAt?: Date;
    afterEnqueuedAt?: Date;
    beforeStartedAt?: Date;
    afterStartedAt?: Date;
    beforeFinishedAt?: Date;
    afterFinishedAt?: Date;
    limit?: number;
    from?: number;
};
export type CancelTasksQuery = Omit<TasksQuery, 'limit' | 'from'> & {};
export type DeleteTasksQuery = Omit<TasksQuery, 'limit' | 'from'> & {};
export type EnqueuedTaskObject = {
    taskUid: number;
    indexUid?: string;
    status: TaskStatus;
    type: TaskTypes;
    enqueuedAt: string;
    canceledBy: number;
};
export type TaskObject = Omit<EnqueuedTaskObject, 'taskUid'> & {
    uid: number;
    details: {
        receivedDocuments?: number;
        indexedDocuments?: number;
        deletedDocuments?: number;
        providedIds?: number;
        primaryKey?: string;
        rankingRules?: RankingRules;
        searchableAttributes?: SearchableAttributes;
        displayedAttributes?: DisplayedAttributes;
        filterableAttributes?: FilterableAttributes;
        sortableAttributes?: SortableAttributes;
        stopWords?: StopWords;
        synonyms?: Synonyms;
        distinctAttribute?: DistinctAttribute;
        swaps?: SwapIndexesParams;
        matchedTasks?: number;
        canceledTasks?: number;
        deletedTasks?: number;
        originalFilter?: string;
    };
    error: MeiliSearchErrorInfo | null;
    duration: string;
    startedAt: string;
    finishedAt: string;
};
export type SwapIndexesParams = Array<{
    indexes: string[];
}>;
type CursorResults<T> = {
    results: T[];
    limit: number;
    from: number;
    next: number;
    total: number;
};
export type TasksResults = CursorResults<Task>;
export type TasksResultsObject = CursorResults<TaskObject>;
export type WaitOptions = {
    timeOutMs?: number;
    intervalMs?: number;
};
export type Health = {
    status: 'available';
};
export type IndexStats = {
    numberOfDocuments: number;
    isIndexing: boolean;
    fieldDistribution: FieldDistribution;
};
export type Stats = {
    databaseSize: number;
    lastUpdate: string;
    indexes: {
        [index: string]: IndexStats;
    };
};
export type Key = {
    uid: string;
    description: string;
    name: string | null;
    key: string;
    actions: string[];
    indexes: string[];
    expiresAt: Date;
    createdAt: Date;
    updatedAt: Date;
};
export type KeyCreation = {
    uid?: string;
    name?: string;
    description?: string;
    actions: string[];
    indexes: string[];
    expiresAt: Date | null;
};
export type KeyUpdate = {
    name?: string;
    description?: string;
};
export type KeysQuery = ResourceQuery & {};
export type KeysResults = ResourceResults<Key[]> & {};
export type Version = {
    commitSha: string;
    commitDate: string;
    pkgVersion: string;
};
export interface FetchError extends Error {
    type: string;
    errno: string;
    code: string;
}
export type MeiliSearchErrorInfo = {
    code: string;
    link: string;
    message: string;
    type: string;
};
export declare const ErrorStatusCode: {
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#index_creation_failed */
    INDEX_CREATION_FAILED: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_index_uid */
    MISSING_INDEX_UID: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#index_already_exists */
    INDEX_ALREADY_EXISTS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#index_not_found */
    INDEX_NOT_FOUND: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_index_uid */
    INVALID_INDEX_UID: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#index_not_accessible */
    INDEX_NOT_ACCESSIBLE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_index_offset */
    INVALID_INDEX_OFFSET: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_index_limit */
    INVALID_INDEX_LIMIT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_state */
    INVALID_STATE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#primary_key_inference_failed */
    PRIMARY_KEY_INFERENCE_FAILED: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#index_primary_key_already_exists */
    INDEX_PRIMARY_KEY_ALREADY_EXISTS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_index_primary_key */
    INVALID_INDEX_PRIMARY_KEY: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#max_fields_limit_exceeded */
    DOCUMENTS_FIELDS_LIMIT_REACHED: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_document_id */
    MISSING_DOCUMENT_ID: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_document_id */
    INVALID_DOCUMENT_ID: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_content_type */
    INVALID_CONTENT_TYPE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_content_type */
    MISSING_CONTENT_TYPE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_document_fields */
    INVALID_DOCUMENT_FIELDS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_document_limit */
    INVALID_DOCUMENT_LIMIT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_document_offset */
    INVALID_DOCUMENT_OFFSET: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_document_filter */
    INVALID_DOCUMENT_FILTER: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_document_filter */
    MISSING_DOCUMENT_FILTER: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_document_vectors_field */
    INVALID_DOCUMENT_VECTORS_FIELD: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#payload_too_large */
    PAYLOAD_TOO_LARGE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_payload */
    MISSING_PAYLOAD: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#malformed_payload */
    MALFORMED_PAYLOAD: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#no_space_left_on_device */
    NO_SPACE_LEFT_ON_DEVICE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_store_file */
    INVALID_STORE_FILE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_ranking_rules */
    INVALID_RANKING_RULES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_request */
    INVALID_REQUEST: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_document_geo_field */
    INVALID_DOCUMENT_GEO_FIELD: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_q */
    INVALID_SEARCH_Q: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_offset */
    INVALID_SEARCH_OFFSET: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_limit */
    INVALID_SEARCH_LIMIT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_page */
    INVALID_SEARCH_PAGE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_hits_per_page */
    INVALID_SEARCH_HITS_PER_PAGE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_attributes_to_retrieve */
    INVALID_SEARCH_ATTRIBUTES_TO_RETRIEVE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_attributes_to_crop */
    INVALID_SEARCH_ATTRIBUTES_TO_CROP: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_crop_length */
    INVALID_SEARCH_CROP_LENGTH: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_attributes_to_highlight */
    INVALID_SEARCH_ATTRIBUTES_TO_HIGHLIGHT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_show_matches_position */
    INVALID_SEARCH_SHOW_MATCHES_POSITION: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_filter */
    INVALID_SEARCH_FILTER: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_sort */
    INVALID_SEARCH_SORT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_facets */
    INVALID_SEARCH_FACETS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_highlight_pre_tag */
    INVALID_SEARCH_HIGHLIGHT_PRE_TAG: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_highlight_post_tag */
    INVALID_SEARCH_HIGHLIGHT_POST_TAG: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_crop_marker */
    INVALID_SEARCH_CROP_MARKER: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_matching_strategy */
    INVALID_SEARCH_MATCHING_STRATEGY: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_vector */
    INVALID_SEARCH_VECTOR: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_search_attributes_to_search_on */
    INVALID_SEARCH_ATTRIBUTES_TO_SEARCH_ON: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#bad_request */
    BAD_REQUEST: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#document_not_found */
    DOCUMENT_NOT_FOUND: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#internal */
    INTERNAL: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_api_key */
    INVALID_API_KEY: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_api_key_description */
    INVALID_API_KEY_DESCRIPTION: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_api_key_actions */
    INVALID_API_KEY_ACTIONS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_api_key_indexes */
    INVALID_API_KEY_INDEXES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_api_key_expires_at */
    INVALID_API_KEY_EXPIRES_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#api_key_not_found */
    API_KEY_NOT_FOUND: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#immutable_api_key_uid */
    IMMUTABLE_API_KEY_UID: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#immutable_api_key_actions */
    IMMUTABLE_API_KEY_ACTIONS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#immutable_api_key_indexes */
    IMMUTABLE_API_KEY_INDEXES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#immutable_api_key_expires_at */
    IMMUTABLE_API_KEY_EXPIRES_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#immutable_api_key_created_at */
    IMMUTABLE_API_KEY_CREATED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#immutable_api_key_updated_at */
    IMMUTABLE_API_KEY_UPDATED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_authorization_header */
    MISSING_AUTHORIZATION_HEADER: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#unretrievable_document */
    UNRETRIEVABLE_DOCUMENT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#database_size_limit_reached */
    MAX_DATABASE_SIZE_LIMIT_REACHED: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#task_not_found */
    TASK_NOT_FOUND: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#dump_process_failed */
    DUMP_PROCESS_FAILED: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#dump_not_found */
    DUMP_NOT_FOUND: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_swap_duplicate_index_found */
    INVALID_SWAP_DUPLICATE_INDEX_FOUND: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_swap_indexes */
    INVALID_SWAP_INDEXES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_swap_indexes */
    MISSING_SWAP_INDEXES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_master_key */
    MISSING_MASTER_KEY: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_types */
    INVALID_TASK_TYPES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_uids */
    INVALID_TASK_UIDS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_statuses */
    INVALID_TASK_STATUSES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_limit */
    INVALID_TASK_LIMIT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_from */
    INVALID_TASK_FROM: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_canceled_by */
    INVALID_TASK_CANCELED_BY: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_task_filters */
    MISSING_TASK_FILTERS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#too_many_open_files */
    TOO_MANY_OPEN_FILES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#io_error */
    IO_ERROR: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_index_uids */
    INVALID_TASK_INDEX_UIDS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#immutable_index_uid */
    IMMUTABLE_INDEX_UID: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#immutable_index_created_at */
    IMMUTABLE_INDEX_CREATED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#immutable_index_updated_at */
    IMMUTABLE_INDEX_UPDATED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_displayed_attributes */
    INVALID_SETTINGS_DISPLAYED_ATTRIBUTES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_searchable_attributes */
    INVALID_SETTINGS_SEARCHABLE_ATTRIBUTES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_filterable_attributes */
    INVALID_SETTINGS_FILTERABLE_ATTRIBUTES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_sortable_attributes */
    INVALID_SETTINGS_SORTABLE_ATTRIBUTES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_ranking_rules */
    INVALID_SETTINGS_RANKING_RULES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_stop_words */
    INVALID_SETTINGS_STOP_WORDS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_synonyms */
    INVALID_SETTINGS_SYNONYMS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_distinct_attribute */
    INVALID_SETTINGS_DISTINCT_ATTRIBUTE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_typo_tolerance */
    INVALID_SETTINGS_TYPO_TOLERANCE: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_faceting */
    INVALID_SETTINGS_FACETING: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_pagination */
    INVALID_SETTINGS_PAGINATION: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_settings_search_cutoff_ms */
    INVALID_SETTINGS_SEARCH_CUTOFF_MS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_before_enqueued_at */
    INVALID_TASK_BEFORE_ENQUEUED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_after_enqueued_at */
    INVALID_TASK_AFTER_ENQUEUED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_before_started_at */
    INVALID_TASK_BEFORE_STARTED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_after_started_at */
    INVALID_TASK_AFTER_STARTED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_before_finished_at */
    INVALID_TASK_BEFORE_FINISHED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_task_after_finished_at */
    INVALID_TASK_AFTER_FINISHED_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_api_key_actions */
    MISSING_API_KEY_ACTIONS: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_api_key_indexes */
    MISSING_API_KEY_INDEXES: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_api_key_expires_at */
    MISSING_API_KEY_EXPIRES_AT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_api_key_limit */
    INVALID_API_KEY_LIMIT: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_api_key_offset */
    INVALID_API_KEY_OFFSET: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_facet_search_facet_name */
    INVALID_FACET_SEARCH_FACET_NAME: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#missing_facet_search_facet_name */
    MISSING_FACET_SEARCH_FACET_NAME: string;
    /** @see https://www.meilisearch.com/docs/reference/errors/error_codes#invalid_facet_search_facet_query */
    INVALID_FACET_SEARCH_FACET_QUERY: string;
};
export type ErrorStatusCode = (typeof ErrorStatusCode)[keyof typeof ErrorStatusCode];
export type TokenIndexRules = {
    [field: string]: any;
    filter?: Filter;
};
export type TokenSearchRules = Record<string, TokenIndexRules | null> | string[];
export type TokenOptions = {
    apiKey?: string;
    expiresAt?: Date;
};
export {};
//# sourceMappingURL=types.d.ts.map